"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOneUser = exports.changeUserRole = exports.deleteUser = exports.updateUser = exports.getTaxAndShipping = exports.getAllUsers = exports.addToCart = exports.getCartItemCount = exports.removeCartItem = exports.getUserCartItem = exports.getUserAccountDetails = void 0;
const db_1 = __importDefault(require("../DB/db"));
const asyncHandler_1 = __importDefault(require("../utils/asyncHandler"));
const apiError_1 = __importDefault(require("../utils/apiError"));
const apiResponse_1 = __importDefault(require("../utils/apiResponse"));
const r2_1 = require("../utils/r2");
// get User Account Details
const getUserAccountDetails = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    if (!id) {
        throw new apiError_1.default(false, 400, 'User id is required');
    }
    const alluserOrder = yield db_1.default.order.findMany({
        where: {
            userId: id,
        },
        include: {
            user: true,
            shippingAddress: true,
            items: true,
            payment: true,
        },
    });
    const getAllShippingAddress = yield db_1.default.address.findMany({
        where: {
            userId: id,
        },
        include: {
            user: true,
        },
    });
    const getPaymentMethod = yield db_1.default.payment.findMany({
        where: {
            order: {
                userId: id,
            },
        },
        include: {
            order: true,
        },
    });
    const userProfile = yield db_1.default.user.findUnique({
        where: {
            id,
        },
    });
    return res.status(200).json(new apiResponse_1.default(true, 200, 'Fetched all users Account Details successfully', {
        alluserOrder,
        getAllShippingAddress,
        getPaymentMethod,
        userProfile,
    }));
}));
exports.getUserAccountDetails = getUserAccountDetails;
// remove cart Items
const removeCartItem = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { itemId } = req.params;
    if (!itemId) {
        throw new apiError_1.default(false, 400, 'User id is required');
    }
    const cartItem = yield db_1.default.cartItem.delete({
        where: {
            id: itemId,
        },
    });
    return res.status(200).json(new apiResponse_1.default(true, 200, ' Cart Items Removed successfully'));
}));
exports.removeCartItem = removeCartItem;
// Fetch all users
const getAllUsers = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const users = yield db_1.default.$queryRaw `
  SELECT 
    u.id, u."fullName", u.email, u.phone, u.role,
    u."countryId", c."countryName" AS "countryName",
    u."provinceId", u."avatarUrl", u.address,
    u."isActive", u."createdAt", u."updatedAt",
    lm."latestMessageDate"
  FROM "User" u
  LEFT JOIN "Country" c
    ON c.id = u."countryId"
  LEFT JOIN (
    SELECT "toUserId", MAX("createdAt") AS "latestMessageDate"
    FROM "Message"
    GROUP BY "toUserId"
  ) lm
    ON lm."toUserId" = u.id
 ORDER BY lm."latestMessageDate" ASC NULLS LAST;

`;
    return res.status(200).json(new apiResponse_1.default(true, 200, 'Fetched all users successfully', users));
}));
exports.getAllUsers = getAllUsers;
// Update user info
const updateUser = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    const { id } = req.params;
    const { fullName, phone, address, countryId, provinceId } = req.body;
    const avatar = (_a = req.file) === null || _a === void 0 ? void 0 : _a.path;
    const user = yield db_1.default.user.findUnique({ where: { id } });
    if (!user)
        throw new apiError_1.default(false, 404, 'User not found');
    let avatarUrl = user.avatarUrl;
    if (avatar) {
        const uploadedUrl = yield (0, r2_1.uploadToCloudinary)(avatar);
        if (!uploadedUrl)
            throw new apiError_1.default(false, 500, 'Avatar upload failed');
        avatarUrl = uploadedUrl;
    }
    const updatedUser = yield db_1.default.user.update({
        where: { id },
        data: {
            fullName: fullName !== null && fullName !== void 0 ? fullName : user.fullName,
            phone: phone !== null && phone !== void 0 ? phone : user.phone,
            address: address !== null && address !== void 0 ? address : user.address,
            countryId: countryId !== null && countryId !== void 0 ? countryId : user.countryId,
            provinceId: provinceId !== null && provinceId !== void 0 ? provinceId : user.provinceId,
            avatarUrl,
        },
    });
    return res.status(200).json(new apiResponse_1.default(true, 200, 'User updated successfully', updatedUser));
}));
exports.updateUser = updateUser;
// Delete user
const deleteUser = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const user = yield db_1.default.user.findUnique({ where: { id } });
    if (!user)
        throw new apiError_1.default(false, 404, 'User not found');
    yield db_1.default.user.delete({ where: { id } });
    return res.status(200).json(new apiResponse_1.default(true, 200, 'User deleted successfully', null));
}));
exports.deleteUser = deleteUser;
// Change user role
const changeUserRole = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const { role } = req.body;
    // Validate role
    const validRoles = [
        'chairman',
        'country_manager',
        'finance',
        'legal',
        'hr',
        'admin',
        'it',
        'councilor',
        'volunteer',
        'viewer',
    ];
    if (!role || !validRoles.includes(role)) {
        throw new apiError_1.default(false, 400, 'Invalid role provided');
    }
    const user = yield db_1.default.user.findUnique({ where: { id } });
    if (!user)
        throw new apiError_1.default(false, 404, 'User not found');
    const updatedUser = yield db_1.default.user.update({
        where: { id },
        data: { role },
    });
    return res
        .status(200)
        .json(new apiResponse_1.default(true, 200, 'User role updated successfully', updatedUser));
}));
exports.changeUserRole = changeUserRole;
// get only one user by id
const getOneUser = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    if (!id) {
        throw new apiError_1.default(false, 401, 'user must login');
    }
    const user = yield db_1.default.user.findUnique({
        where: { id },
        select: {
            fullName: true,
            email: true,
            id: true,
            avatarUrl: true,
            country: true,
            role: true,
        },
    });
    return res.status(200).json(new apiResponse_1.default(true, 200, 'User get successfully', user));
}));
exports.getOneUser = getOneUser;
// get tax and shipping cost
const getTaxAndShipping = (0, asyncHandler_1.default)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const getData = yield db_1.default.storeSettings.findFirst({
        select: {
            taxRate: true,
            freeShipping: true,
        }
    });
    return res.status(200).json(new apiResponse_1.default(true, 200, 'Success', getData));
}));
exports.getTaxAndShipping = getTaxAndShipping;
